package com.mini.framework.util.page;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import com.mini.framework.core.status.Status;
import com.mini.framework.core.status.Status.Busy;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;

/**
 * 查询排序器
 * 用于处理排序条件
 * @author jayheo
 *
 */
public class QuerySorter {

	public static final String emptyAlert = "需要排序参数sorterAsKey和sorterAsDesc"; 
	
	/**
	 * <pre>
	 * 排序的key
	 * 如果通过前端传第一个这样的key可使用逗号隔开的形式。
	 * 例如key1,key2
	 * </pre>
	 */
	private String[] sorterAsKey;
	
	/**
	 * <pre>
	 * 是否要 倒序
	 * 如果通过前端传第一个这样的key可使用逗号隔开的形式。
	 * 例如true,false
	 * 以上表示第一个要倒序，第二个要逆序
	 * </pre>
	 */
	private Boolean[] sorterAsDesc;
	
	
	public Optional<Map<String, Boolean>> validate(){
		if(sorterAsDesc==null){
			AssertUtil.assertNoBadReq(sorterAsKey==null,Status.BadReq.illParam,"sorterAsDesc[null]与sorterAsKey:%s不能一个为空一个不是空",Arrays.toString(sorterAsKey));
			return Optional.empty();
		}else{
			AssertUtil.assertNoBadReq(sorterAsKey!=null,Status.BadReq.illParam,"sorterAsDesc%s与sorterAsKey:[null]不能一个为空一个不是空",Arrays.toString(sorterAsDesc));
			AssertUtil.assertNoBadReq(sorterAsKey.length == sorterAsDesc.length,Status.BadReq.illParam,"sorterAsDesc%s与sorterAsKey:[%s]长度必须一致",Arrays.toString(sorterAsDesc),Arrays.toString(sorterAsKey));
			LinkedHashMap<String, Boolean> sorter = new LinkedHashMap<>();
			for (int i = 0; i < sorterAsDesc.length; i++) {
				String key = sorterAsKey[i];
				Boolean desc = sorterAsDesc[i];
				AssertUtil.assertNoBadReq(key!=null && key.length()>0, Status.BadReq.illParam,"sorterAsKey:[%s]第[%s]个不能为空:[%s]",sorterAsKey, i+1,key);
				AssertUtil.assertNoBadReq(desc!=null , Status.BadReq.illParam,"sorterAsDesc:[%s]第[%s]个不能为空:[%s]",sorterAsKey, i+1,desc);
				AssertUtil.assertNoBadReq(!sorter.containsKey(key) , Status.BadReq.illParam,"sorterAsDesc:[%s]第[%s]个key:[%s]出现重复",sorterAsKey, i+1,desc);
				sorter.put(key, desc);
			}
			
			return Optional.of(sorter);
			
		}
	}
	

	/**
	 * 获取填充排序器的过滤
	 * @return
	 */
	public Consumer< MapParams> fillSorterProcess(){
		return fillSorterProcess(Function.identity());
	}
	
	/**
	 * 获取填充排序器的过滤
	 * @param fieldGet
	 * @return
	 */
	public Consumer< MapParams> fillSorterProcess(Function<String, String> fieldGet){
		Optional<Map<String, Boolean>> sorterOptional = validate();
		return mapParams ->  {
			if(sorterOptional.isPresent()){
				for (Entry<String, Boolean> entry : sorterOptional.get().entrySet()) {
					String realKey = fieldGet.apply(entry.getKey());
					//如果realKey为空应该一定会报错，所以要提前阻止
					AssertUtil.assertNotFatal(realKey!=null, Busy.bug, "fieldGet.apply(%s)为空", entry.getKey());
					mapParams.orderBy(realKey, entry.getValue());
				}
			}
		};
	}
	

	public String[] getSorterAsKey() {
		return sorterAsKey;
	}

	public void setSorterAsKey(String[] sorterAsKey) {
		this.sorterAsKey = sorterAsKey;
	}

	public Boolean[] getSorterAsDesc() {
		return sorterAsDesc;
	}

	public void setSorterAsDesc(Boolean[] sorterAsDesc) {
		this.sorterAsDesc = sorterAsDesc;
	}
	
	
}
