package com.mini.framework.util.string.gson;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;


import java.util.stream.Collectors;
import java.util.stream.Stream;


import org.apache.commons.lang3.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;




import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.string.GsonUtil;

/**
 * 通过gson读取数据
 * @author jayheo
 *
 */
public class GsonDataReader {
	
	private static Logger logger = LogManager.getLogger(GsonDataReader.class);
	
	protected Object origin;
	
	protected JsonElement currObject;
 	
	public String createString(){
		return currObject.toString();
	}
	public String toString(){
		return currObject.toString();
	}

	public Object getOrigin() {
		return origin;
	}

	public void setOrigin(Object origin) {
		this.origin = origin;
	}
	
	public void init(Object object){
		this.setOrigin(object);
		this.getCurr();
	}
	
	protected  Gson currGsonUtil(){
		return GsonUtil.buildMilliSecondDateGson();
	}
	
	protected JsonElement getCurr(){
		if(currObject==null){
			//TODO null判断
			// 这里注意，如果出现这样的，会导致内存溢出	private Logger logger = LogManager.getLogger();
			//后面要允许可以修改json的转化方式
			currObject = currGsonUtil().toJsonTree(origin);
		}
		return currObject;
	}
	

	/**
	 * 根据命名空间取json元素中的json对象
	 * @param jsonElement
	 * @param names
	 * @return
	 */
	protected List<JsonObject> getObject(JsonElement jsonElement,String... names){
		AssertUtil.assertMethodRequire(jsonElement, "jsonElement");
		AssertUtil.assertSupport(names!=null && names.length>=0, "names长度要大于1");
		List<JsonObject> elements = new ArrayList<JsonObject>();
		if(jsonElement.isJsonNull()){//如果没有这个json
			return elements;
		}else if(jsonElement.isJsonObject()){//如果是一个json对象
			if(names.length==0){
				elements.add(jsonElement.getAsJsonObject());
			}else{
				JsonElement subJson = jsonElement.getAsJsonObject().get(names[0]);
				if (subJson!=null){//如果没有这个数据就忽略
					elements.addAll(getObject(subJson,ArrayUtils.subarray(names, 1, names.length)));
				}
			}
		}else if(jsonElement.isJsonArray()){
			for (JsonElement subJsonElement : jsonElement.getAsJsonArray()) {
				elements.addAll(getObject(subJsonElement,names));
			}
		}else{
			logger.debug("发现一个没有处理的jsonElement, names:{},jsonElement:{}",jsonElement,Arrays.toString(names));
		}
		return elements;
	}


	protected List<Map<String, Object>> getFromListFrame(String listNamePath,String... elementKeys) {
		return getFromListFrame(listNamePath, Stream.of(elementKeys).collect(Collectors.toList()));
	}
	
	protected List<Map<String, Object>> getFromListFrame(String listNamePath,List<String> elementKeys) {
		List<JsonObject> list = getObject(listNamePath);
		return list.stream().map(jsonElement->{
			Map<String, Object> map = new HashMap<>();
			for (String namespace : elementKeys) {
				Object value = getSingleValue(jsonElement, namespace).orElse(null);
				map.put(namespace, value);
			}
			return map;
		}).collect(Collectors.toList());
		
		
	}
	
	
	/**
	 * 直接取出对象
	 * @param jsonElement
	 * @param namespace 他对应的结果必须是对象中间不能出现数组
	 * @return
	 */
	protected Optional<Object> getSingleValue(JsonElement jsonElement,String namespace){
		String[] names = namespace.split("\\.");
		return getSingleValue(jsonElement, names);
		
	}
	
	
	/**
	 * 直接取出对象
	 * @param jsonElement
	 * @param names 他对应的结果必须是对象中间不能出现数组
	 * @return
	 */
	protected Optional<Object> getSingleValue(JsonElement jsonElement,String... names){
		JsonElement resultTemp = jsonElement;
		for (String name : names) {
			if(resultTemp==null){
				return Optional.empty();
			}else if(resultTemp.isJsonNull()){
				return Optional.empty();
			}else if(resultTemp.isJsonPrimitive()){
				return Optional.empty();
			}else if(resultTemp.isJsonArray()){
				throw new BadReqException("命名空间:[%s]错误,它得到的数据是数组", String.join(".", names));
			}else if(resultTemp.isJsonObject()){
				resultTemp = resultTemp.getAsJsonObject().get(name);
			}else{
				logger.error("从jsonElement:{}中取:{}出现未知的json类型",jsonElement,String.join(".", names));
			}
		}
		return Optional.ofNullable(resultTemp).map(JsonElement::getAsJsonPrimitive).map(JsonPrimitive::getAsString);
	}
	

	protected List<JsonObject> getObject(String... names){
		JsonElement element = getCurr();
		return getObject(element, names);
	}
	
	/**
	 * 根据命名空间取对应的json对象，如果碰到数组直接拆分成多个
	 * @param namespace
	 * @return
	 */
	protected List<JsonObject> getObject(String namespace){
		String[] names = namespace.split("\\.");
		if(names[0].length()==0){
			names = new String[]{};
		}
		return getObject(names);
	}
	
}
