package com.mini.framework.third.weixin.server.client;

import java.util.Optional;
import java.util.function.Consumer;

import com.mini.framework.third.weixin.server.comm.AbstractResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.google.gson.Gson;
import com.mini.framework.third.weixin.WeixinAppidType;
import com.mini.framework.third.weixin.server.comm.AccessTokenResponse;
import com.mini.framework.third.weixin.server.comm.CreateTemplateResponse;
import com.mini.framework.third.weixin.server.comm.DefaultMenuResonpse;
import com.mini.framework.third.weixin.server.comm.OpenidListResponse;
import com.mini.framework.third.weixin.server.comm.SceneQrcodeRequest;
import com.mini.framework.third.weixin.server.comm.SceneQrcodeResponse;
import com.mini.framework.third.weixin.server.comm.SimpleResponse;
import com.mini.framework.third.weixin.server.comm.TempMsgResponse;
import com.mini.framework.third.weixin.server.comm.TemplateListResponse;
import com.mini.framework.third.weixin.server.comm.TicketResponse;
import com.mini.framework.third.weixin.server.model.ServerJsapiTicketResponse;
import com.mini.framework.third.weixin.server.model.WeixinServerAccountAccess;
import com.mini.framework.third.weixin.server.model.WeixinServerUserDetail;
import com.mini.framework.third.weixin.server.model.WeixinWebAuthTokenResp;
import com.mini.framework.third.weixin.server.provider.WeixinServerAccessProvider;
import com.mini.framework.third.weixin.server.provider.WeixinSiteTokenProvider;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.http.SimpleHttpUtil;
import com.mini.framework.util.string.GsonUtil;

@Component
public class WeixinServerSdkSimpleImpl implements WeixinServerAccountSdk{
	//private String server = "https://api.weixin.qq.com";
	
	public static final String defaultWeixinApiHost = "https://api.weixin.qq.com";
	
	@Value("${weixin.open.api.url:" +defaultWeixinApiHost+ "}")
	private String server;
	
	private String moudle = "/cgi-bin";
	
	@Autowired
	private WeixinSiteTokenProvider siteTokenProvider; 
	
	@Autowired
	private WeixinServerAccessProvider weixinServerAccessProvider;
	
	private Logger logger = LogManager.getLogger(WeixinServerSdkSimpleImpl.class);
	
	
	public WeixinSiteTokenProvider getSiteTokenProvider() {
		return siteTokenProvider;
	}
	public void setSiteTokenProvider(WeixinSiteTokenProvider siteTokenProvider) {
		this.siteTokenProvider = siteTokenProvider;
	}
	
	public WeixinServerAccessProvider getWeixinServerAccessProvider() {
		return weixinServerAccessProvider;
	}
	public void setWeixinServerAccessProvider(WeixinServerAccessProvider weixinServerAccessProvider) {
		this.weixinServerAccessProvider = weixinServerAccessProvider;
	}
	public String getServer() {
		return server;
	}
	public void setServer(String server) {
		this.server = server;
	}
	
	private String currToken(){
		return siteTokenProvider.currSiteToken(WeixinAppidType.server);
	}


	/**
	 * 获取一个明确有的server值。
	 * 这么写的目的是在缺少参数时及时直到。
	 * @return
	 */
	private String getHadServer(){
		AssertUtil.assertNotFatalBug(server!=null,"微信服务号sdk中参数server不能为空");
		return server;
	}
	
	@Override
	public SceneQrcodeResponse createSceneQrcode(SceneQrcodeRequest req) {
		String token = currToken();
		AssertUtil.assertNotFatal(token!=null,"微信token不能为空");
		String action = "/qrcode/create?access_token=" + currToken();
		String url = getHadServer()  + moudle + action ;
		String reqBody  = req.serialize();
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SceneQrcodeResponse resp = new Gson().fromJson(respStr, SceneQrcodeResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}
	@Override
	public AccessTokenResponse createAccessToken() {
		String action = "/token?grant_type=client_credential&appid=%s&secret=%s";
		 WeixinServerAccountAccess access = weixinServerAccessProvider.serverProvide();
		String appid = access.getAppid();
		String secret =access.getSecret();
		String url = getHadServer()  + moudle + String.format(action ,appid,secret);
		String respStr = SimpleHttpUtil.simpleGet(url);
		AccessTokenResponse resp = new Gson().fromJson(respStr, AccessTokenResponse.class);
		resp.handle("url:%s",url);
		return resp;
	}
	
	@Override
	public WeixinServerUserDetail queryUserDetailByOpenid(String openid) {
		String action = "/user/info?access_token=%s&openid=%s&lang=zh_CN";
		String token = currToken();
		String url = getHadServer()  + moudle + String.format(action ,token,openid);
		String respStr = SimpleHttpUtil.simpleGet(url);
		WeixinServerUserDetail resp = WeixinServerUserDetail.fromJson( WeixinServerUserDetail.class,respStr);
		resp.handle("url:%s",url);
		return resp;
	}
	
	
	@Override
	public WeixinServerUserDetail queryAuthByCode( String code,Consumer<WeixinWebAuthTokenResp> tokenRespConsumer) {
		String action = "/sns/oauth2/access_token?appid=%s&secret=%s&code=%s&grant_type=authorization_code";
		 WeixinServerAccountAccess access = weixinServerAccessProvider.serverProvide();
		String appid = access.getAppid();
		String secret =access.getSecret();
		//这不是cgi的模块
		String url = getHadServer()  + String.format(action ,appid,secret,code);
		String respStr = SimpleHttpUtil.simpleGet(url);
		WeixinWebAuthTokenResp tokenResp = GsonUtil.buildSecondDateGson().fromJson(respStr, WeixinWebAuthTokenResp.class);
		tokenResp.handle("url:%s",url);
		tokenRespConsumer.accept(tokenResp);
        if(tokenResp.haveUserInfoModel()){//48001,"errmsg":"api unauthorized  如果scope不是使用 UserInfoModel那么 会报出这个的错误。但是好像只曾经授权过短时间还是有权限。
        	return queryUserDetailByUserAccessToken(tokenResp.getAccessToken(),tokenResp.getOpenId());
        }else{//48001,"errmsg":"api unauthorized 
        	WeixinServerUserDetail simple = new WeixinServerUserDetail();
        	simple.setOpenId(tokenResp.getOpenId());
        	return simple;
        }
	}
	@Override
	public WeixinServerUserDetail queryUserDetailByUserAccessToken(String accessToken, String openid) {
		String action = "/sns/userinfo?access_token=%s&openid=%s&lang=zh_CN";
		String url = getHadServer()  + String.format(action ,accessToken,openid);
		String respStr = SimpleHttpUtil.simpleGet(url);
		WeixinServerUserDetail resp = WeixinServerUserDetail.fromJson( WeixinServerUserDetail.class,respStr);
		resp.handle("url:%s",url);
		return resp;
	}
	
	@Override
	public void sendTextMsg(String openId, String content) {
		String token = currToken();
		String action = "/message/custom/send?access_token=%s";
		String bodyTemp = "{\"touser\": \"%s\", \"msgtype\": \"text\", \"text\": {\"content\": \"%s\"}}";
		String body = String.format(bodyTemp, openId,content);
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simplePost(url, body);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",body);
	}
	@Override
	public TicketResponse createJsapiTicketResponse() {
		String token = currToken();
		String action = "/ticket/getticket?access_token=%s&type=jsapi";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simpleGet(url);
		TicketResponse resp = AbstractResponse.fromJson(TicketResponse.class,respStr);
		resp.handle("url:%s",url);
		//如果expire不是7200秒的时候就要注意了。
		return resp;
	}
	
	public ServerJsapiTicketResponse createJsapiTicket(String url){
		String ticket = createJsapiTicket();
		
		WeixinServerAccountAccess access = weixinServerAccessProvider.serverProvide();
		String appid = access.getAppid();
		ServerJsapiTicketResponse ticketResp = new ServerJsapiTicketResponse(url, ticket, appid);
		ticketResp.sign();
		return ticketResp;
	}
	
	@Override
	public String sendTempateMsg(String reqBody) {
		AssertUtil.assertMethodRequire(reqBody, "reqBody");
		String token = currToken();
		//解决微信中\n的问题，解析不了的问题。
		reqBody = reqBody.replaceAll("\n", "\\\\n");
		String action = "/message/template/send?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simplePost(url,reqBody);
		TempMsgResponse resp = new Gson().fromJson(respStr, TempMsgResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp.getMessageId();
	}
	@Override
	public String createMsgTempate(String templateKey) {
		AssertUtil.assertMethodRequire(templateKey, "templateKey");
		String token = currToken();
		String action = "/template/api_add_template?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String reqBody = String.format("{\"template_id_short\":\"%s\"}", templateKey);
		String respStr = SimpleHttpUtil.simplePost(url,reqBody );
		CreateTemplateResponse resp = new Gson().fromJson(respStr, CreateTemplateResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp.getTemplateId();
	}
	@Override
	public TemplateListResponse queryExistTemplateList() {
		String token = currToken();
		String action = "/template/get_all_private_template?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simpleGet(url);
		TemplateListResponse resp = new Gson().fromJson(respStr, TemplateListResponse.class);
		//不需要handle
		resp.handle("url:%s",url);
		return resp;
	}
	@Override
	public void deleteTemplateResponse(String templateId) {
		AssertUtil.assertMethodRequire(templateId, "templateId");
		String token = currToken();
		String action = "/template/del_private_template?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String reqBody = String.format("{\"template_id\":\"%s\"}", templateId);
		String respStr = SimpleHttpUtil.simplePost(url,reqBody );
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",reqBody);
	}
	/**
	 * {
    "primary_industry": {
        "first_class": "教育",
        "second_class": "培训"
    },
    "secondary_industry": {
        "first_class": "商业服务",
        "second_class": "广告|会展"
    }
}
	 * */
	@Override
	public String queryTemplateIndustry() {
		String token = currToken();
		String action = "/template/get_industry?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simpleGet(url);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("url:%s",url);
		return respStr;
	}
	
	
	@Override
	public OpenidListResponse querySubscribeOpenidList(String offset) {
		String token = currToken();
		String action = "/user/get?access_token=%s&next_openid=%s";
		String url = getHadServer()  + moudle + String.format(action ,token,offset==null?"":offset);
		String respStr = SimpleHttpUtil.simpleGet(url);
		OpenidListResponse resp = new Gson().fromJson(respStr, OpenidListResponse.class);
		resp.handle("url:%s",url);
		return resp;
	}
	@Override
	public DefaultMenuResonpse getServerDefaultMenu() {
		String token = currToken();
		String action = "/get_current_selfmenu_info?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simpleGet(url);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("url:%s",url);
		return DefaultMenuResonpse.fromJson(respStr);
	}
	@Override
	public void setServerDefaultMenu(String reqJson) {
		logger.info("修改微信服务号默认菜单报文:{}" ,reqJson);
		String token = currToken();
		String action = "/menu/create?access_token=%s";
		String url = getHadServer()  + moudle + String.format(action ,token);
		String respStr = SimpleHttpUtil.simplePost(url,reqJson);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("url:%s",url);
	}
	@Override
	public Optional<String> currentRunnableServerAppid() {
		WeixinServerAccountAccess access = weixinServerAccessProvider.serverProvide();
		return Optional.ofNullable(access.getAppid());
	}
}
