package com.mini.framework.util.cache;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Predicate;

import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.bean.VarUtil;

/**
 * 冻结一个function，以保证同样的参数，只被执行一次
 * 一般来说， 这个T是要基本数据类型可以被equals的
 * 支持空值
 * @author jayheo
 *
 * @param <T>
 * @param <R>
 */
public class HoldBiFunction<T,U,R> implements BiFunction<T, U, R>{

	private Map<Entry<T,U>,R> cacheMapper = new HashMap<>();
	/**
	 * 标记一下这个有没有被使用，主要是解决null值被多次执行的问题
	 */
	private List<Entry<T,U>> executeFlag = new ArrayList<>();
	
	private BiFunction<T, U, R> biFunction = null;
	
	private HoldBiFunction(BiFunction<T, U, R> biFunction) {
		super();
		this.biFunction = biFunction;
	}

	public static <T,U,R> HoldBiFunction<T,U, R> create(BiFunction<T,U, R> biFunction){
		AssertUtil.assertMethodRequire(biFunction, "biFunction");
		return new HoldBiFunction<>(biFunction);
	}


	@Override
	public R apply(T t,U u) {
		if(containFlagKey(t,u)){
			return getFromCacheMapper(t, u).orElse(null);
		}else{
			addFlagKey(t, u);
			R r = biFunction.apply(t,u);
			cacheMapper.put(VarUtil.createEntry(t, u), r);
			return r;
		}
	}
	
	private Optional<R> getFromCacheMapper(T t,U u){
		entryMatch(t,u);
		return cacheMapper.entrySet().stream().filter(cache->entryMatch(t,u).test(cache.getKey())).findFirst().map(Entry::getValue);
	}
	
	private void addFlagKey(T t,U u){
		executeFlag.add(VarUtil.createEntry(t, u));
	}
	private boolean containFlagKey(T t,U u){
		return executeFlag.stream().anyMatch(entryMatch(t, u));
	}
	
	private Predicate<Entry<T, U>> entryMatch(T t,U u){
		return entry-> varSameStatus(entry.getKey(), t) && varSameStatus(entry.getValue(), u);
	}
	
	
	private <O> boolean varSameStatus(O t1,O t2){
		return (t1==null &&t2==null) || Objects.equals(t1, t2);
	}
	
	
	
	
	public void clear(){
		executeFlag.clear();
		cacheMapper.clear();
	}
}
