package com.mini.framework.util.cache.annotation.springboot2;

import java.time.Duration;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.interceptor.CacheErrorHandler;
import org.springframework.cache.interceptor.SimpleCacheErrorHandler;
import org.springframework.data.redis.cache.CacheKeyPrefix;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheManager.RedisCacheManagerBuilder;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mini.framework.util.log.MiniLogLevel;

/**
 * springboot 2.0 redis支持
 * 支持 注解 
 * @author jayheo
 *
 */
public class AbstractRedisCacheSupport extends CachingConfigurerSupport{
	Logger logger = LogManager.getLogger(AbstractRedisCacheSupport.class);
	
	protected JedisConnectionFactory jedisConnectionFactory;
	
	protected RedisCacheConfiguration defaultRedisCacheConfiguration;
	
	protected String cacheNamespace;
	
	/**
	 * 入口配置
	 * @param redisConnectConfig
	 * @param cacheNamespace
	 */
	public AbstractRedisCacheSupport(RedisConnectConfig redisConnectConfig, String cacheNamespace) {
		this(createJedisConnectionFactory(redisConnectConfig.getHost(), redisConnectConfig.getPort(), redisConnectConfig.getPassword(),redisConnectConfig.getDatabase(), redisConnectConfig.getTimeout()), cacheNamespace);
	}
	
	protected AbstractRedisCacheSupport(){
		
	}
	
	/**
	 * 入口配置
	 * @param jedisConnectionFactory
	 * @param cacheNamespace
	 */
	public AbstractRedisCacheSupport(JedisConnectionFactory jedisConnectionFactory, String cacheNamespace) {
		super();
		afterInit(jedisConnectionFactory, cacheNamespace);
	}
	
	public void afterInit(JedisConnectionFactory jedisConnectionFactory, String cacheNamespace){
		this.jedisConnectionFactory = jedisConnectionFactory;
		this.cacheNamespace = cacheNamespace;
		this.defaultRedisCacheConfiguration = createBaseCacheConfiguration(createKeyPrefix(this.cacheNamespace));
	}

	public static RedisCacheConfiguration createBaseCacheConfiguration(CacheKeyPrefix keyPrefix){
        Jackson2JsonRedisSerializer<Object> jackson2JsonRedisSerializer = new Jackson2JsonRedisSerializer<>(Object.class);
        ObjectMapper om = new ObjectMapper();
        om.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        om.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL);
        jackson2JsonRedisSerializer.setObjectMapper(om);
        RedisSerializationContext.SerializationPair<Object> pair = RedisSerializationContext.SerializationPair
                .fromSerializer(jackson2JsonRedisSerializer);
		RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
    		.computePrefixWith(keyPrefix )
    		// 设置默认缓存过期时间 实际缓存会在AbstractRedisCacheSupport.createRedisCache中重写实现
    		.entryTtl(Duration.ofSeconds(10))
    		.serializeValuesWith(pair)
    		.serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(new StringRedisSerializer()))
    		;
		return config;
	}

	public static JedisConnectionFactory createJedisConnectionFactory(String host,int port,String password,int database,int timeout) {
		RedisStandaloneConfiguration redisStandaloneConfiguration = new RedisStandaloneConfiguration(host, port);
		redisStandaloneConfiguration.setDatabase(database);
		redisStandaloneConfiguration.setPassword(RedisPassword.of(password));
		JedisClientConfiguration jedisClientConfiguration = JedisClientConfiguration
				.builder().connectTimeout(Duration.ofMillis(timeout)).build();// connection timeout
		return new JedisConnectionFactory(redisStandaloneConfiguration,jedisClientConfiguration);
	}
	
	@Override
	public CacheManager cacheManager() {
		CustomCacheKeyPrefix keyPrefix = this.createKeyPrefix(cacheNamespace);
        RedisCacheWriter redisCacheWriter = RedisCacheWriter.nonLockingRedisCacheWriter(jedisConnectionFactory);
		RedisCacheConfiguration defaultCacheConfig = defaultRedisCacheConfiguration;
		RedisCacheManagerBuilder.fromConnectionFactory(jedisConnectionFactory).cacheDefaults(defaultCacheConfig).build().afterPropertiesSet();;
        RedisCacheManager cacheManager = new CustomRedisCacheManager(keyPrefix,jedisConnectionFactory,redisCacheWriter, defaultCacheConfig);
        return cacheManager;
	}
    
	@Override
	public CacheErrorHandler errorHandler() {
		return new SimpleCacheErrorHandler(){
			@Override
			public void handleCacheGetError(RuntimeException exception, Cache cache, Object key) {
				//这个方法是用来处理取换缓存错误情况。
				logger.error("取缓存[{}]时出错",key,exception);
				//TODO 这里错误应该过滤到网络异常，后续改正，只有是在数据错误的情况下才考虑删除数据
				cache.evict(key);
				//TODO 打一点日志，如果错误不多可以忽略  throw exception;
				logger.log(MiniLogLevel.getKeyBizLog(),"忽略错误 并操作删除缓存:[{}]",key);
			}
		};
	}

	protected JedisConnectionFactory showJedisConnectionFactory(){
		return jedisConnectionFactory;
	}
	
	protected CustomCacheKeyPrefix createKeyPrefix(String cacheNamespace){
		return new CustomCacheKeyPrefix(cacheNamespace);
	}
	
}
