package com.mini.framework.util.string.json;

import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.PathNotFoundException;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.string.json.flag.JsonPathMapping;

import java.lang.reflect.Field;
import java.util.LinkedList;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.stream.Stream;


/**
 * json的帮助类
 * 基于jsonPath使用注解来标记路径，对于层次比较深的json解析有很大的帮助。
 */
public class JsonFlagHelper {


    /**
     * 从json字符串中解析需要的对象。
     * @param jsonString
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T parseFrom(String jsonString, Class<T> clazz){
        Supplier<T> instanceSupplier = () -> {
            try {
                return clazz.newInstance();
            } catch (InstantiationException|IllegalAccessException e) {
                throw new ServerException(e, Status.Server.programConfigJava,"创建实例时出错,%s",clazz);
            }
        };

        return parseFromFill(jsonString,instanceSupplier);
    }

    /**
     * 解析字符串并填冲到到某个实例中去
     * 要想被真正填充必须打上标记 JsonPathMapping
     * @param jsonString json 字符串。
     * @param instanceSupplier 提供空实例的方法。
     * @param <T> 最终返回的示例
     * @return
     */
    public static <T> T parseFromFill(String jsonString, Supplier<T> instanceSupplier){
        AssertUtil.assertMethodRequire(jsonString,jsonString);
        T instance = instanceSupplier.get();
        Field[] fields = instance.getClass().getDeclaredFields();
        Stream.of(fields).forEach(field -> {
            JsonPathMapping pathMapping = field.getAnnotation(JsonPathMapping.class);
            if(pathMapping!=null){
                String[] paths = pathMapping.value();
                try {
                    Object value = parseFirstMatch(jsonString, (message, runtimeException) -> new ServerException(runtimeException, Status.Server.programConfigJava, "给属性:[%s]赋值时出错,%s", field.getName(),message), paths);
                    field.setAccessible(true);
                    field.set(instance,value);
                    field.setAccessible(false);
                } catch (IllegalAccessException e) {
                    throw new ServerException(e, Status.Server.programConfigJava,"给属性:[%s]赋值时出错",field.getName());
                }
            }
        });
        return instance;
    }


    /**
     * 匹配第一个正常匹配的。
     * @param jsonString
     * @param exceptionSupplier
     * @param pathExplains
     * @param <T>
     * @param <E>
     * @return
     */
    public static <T,E extends CustomException> T parseFirstMatch(String jsonString, BiFunction<String,RuntimeException,E> exceptionSupplier, String... pathExplains){
        LinkedList<RuntimeException> errors = new LinkedList<>();
        AssertUtil.assertSupport(pathExplains.length>0,"pathExplains 不能不传值");
        return Stream.of(pathExplains).map(explain->{
            try {
                T result = JsonPath.read(jsonString,explain);
                return result;
            } catch (PathNotFoundException e) {
                errors.add(e);
                return null;
            } catch (RuntimeException e){
                //其它错误不应该算到里面应该直接暴露出来
                throw exceptionSupplier.apply(String.format("根据路径:[%s]取值失败", explain),e);
            }
        }).filter(Objects::nonNull).findFirst()
        .orElseThrow(()->{
            String lastExplain = pathExplains[pathExplains.length - 1];
            RuntimeException lastError = errors.getLast();
            return exceptionSupplier.apply(String.format("根据路径:[%s]取值失败", lastExplain),lastError);
        });

    }

}
