package com.mini.framework.util.cache.prepare.merge;

import com.mini.framework.util.cache.prepare.merge.data.PrepareAtomCacheExecutor;
import com.mini.framework.util.cache.prepare.merge.data.RedisKeyGenerator;
import com.mini.framework.util.cache.prepare.merge.data.WorkDelayLock;
import com.mini.framework.util.thread.AbstractAsyncThreadExecutor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.data.redis.support.atomic.RedisAtomicInteger;

import java.time.Duration;
import java.util.Optional;

/**
 * @author jayheo
 */
public class RedisIncrementPrepareCacheAsyncMergeFlow extends IncrementPrepareCacheAsyncMergeFlow{



    private RedisTemplate redisTemplate;
    private int errorMaxDelayMillis;


    private RedisKeyGenerator counterGenerator;
    private RedisKeyGenerator workHolderGenerator;



    /**
     * 创建增量预先缓存再异步合并 工作流
     * @param asyncExecutor 异步执行器
     * @param redisTemplate spring redis模板
     * @param counterGenerator 缓存统计器redis key 生成 器
     * @param workHolderGenerator 单独工作redis key 生成 器
     * @param errorMaxDelayMillis 单独工作线程占用最大时间 毫秒
     */
    public RedisIncrementPrepareCacheAsyncMergeFlow(
            String flowName
            ,AbstractAsyncThreadExecutor asyncExecutor
            ,RedisTemplate redisTemplate
            ,RedisKeyGenerator counterGenerator
            ,RedisKeyGenerator workHolderGenerator
            ,int errorMaxDelayMillis
        ) {
        super(flowName,asyncExecutor
                , createRedisPrepareCacheExecute(redisTemplate,counterGenerator)
                , createRedisLock(redisTemplate,workHolderGenerator,errorMaxDelayMillis));
        this.redisTemplate = redisTemplate;
        this.errorMaxDelayMillis = errorMaxDelayMillis;
        this.counterGenerator = counterGenerator;
        this.workHolderGenerator = workHolderGenerator;
        //TODO 检查数据不能为空
    }

    private static WorkDelayLock createRedisLock(RedisTemplate redisTemplate, RedisKeyGenerator keyGenerator, int errorMaxDelayMillis) {

        return new WorkDelayLock() {
            @Override
            public Optional<String> triesLock(String key,String workerName) {
                //TODO threadName应该显示对应的服务器
                String threadName = Thread.currentThread().getName();
                String redisKey = keyGenerator.create(key);
                // 注意 setIfAbsent 是spring-data-redis里的一个新的方法，老版本没有这个方法
                ValueOperations valueOps = redisTemplate.opsForValue();
                Boolean working = valueOps.setIfAbsent(redisKey,threadName, Duration.ofMillis(errorMaxDelayMillis));
                return working ? Optional.empty() : Optional.ofNullable(valueOps.get(redisKey)).map(Object::toString) ;
            }

            @Override
            public void releaseLock(String key) {
                //TODO 要修改一下redis的配置
                String redisKey = keyGenerator.create(key);
                redisTemplate.opsForList();
                redisTemplate.delete(redisKey);
            }
        };

    }

    private static PrepareAtomCacheExecutor createRedisPrepareCacheExecute(RedisTemplate redisTemplate, RedisKeyGenerator keyGenerator){
        return new PrepareAtomCacheExecutor() {
            @Override
            public int writeIncrement(String key, int increment) {
                String redisKey = keyGenerator.create(key);
                RedisAtomicInteger counter = new RedisAtomicInteger(redisKey, redisTemplate.getConnectionFactory());
                return counter.addAndGet(increment);
            }

            @Override
            public int querySurplus(String key) {
                String redisKey = keyGenerator.create(key);
                RedisAtomicInteger counter = new RedisAtomicInteger(redisKey, redisTemplate.getConnectionFactory());
                return counter.get();
            }

            @Override
            public int cutSurplus(String key) {
                String redisKey = keyGenerator.create(key);
                RedisAtomicInteger counter = new RedisAtomicInteger(redisKey, redisTemplate.getConnectionFactory());
                return counter.getAndSet(0);
            }
        };
    }




}
