package com.mini.framework.util.generic.define;


import com.mini.framework.util.asserts.AssertUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * long的set集合 这种写法是解决泛型带来的部分参数不明确的影响<BR>
 * 等价于<br>
 * @see java.util.Set<Long>
 * @author jayheo
 */
public class SetLong {

    /**
     * 初始值，不可以被修改的。
     */
    private final Set<Long> value;

    public SetLong(Set<Long> value) {
        this.value = value;
    }


    public static SetLong empty(){
        return of(new HashSet<>());
    }


    /**
     * 创建一个集合
     * @param longsNullable long数组 可以为空
     * @return SetLong
     */
    public static SetLong ofNullable(Long[] longsNullable){
        return longsNullable==null?empty():of(longsNullable);
    }

    /**
     * 创建一个集合
     * @param setNullable 集合 可以为空
     * @return SetLong
     */
    public static SetLong ofNullable(Set<Long> setNullable){
        return setNullable==null?empty():of(setNullable);
    }


    public static SetLong of(Long... longs){
        return of(Stream.of(longs).collect(Collectors.toSet()));
    }

    public static SetLong of(SetLong setLong1,SetLong setLong2,SetLong setLong3,SetLong setLong4){
        return SetLong.of(SetLong.of(setLong1,setLong2),setLong3,setLong4);
    }

    public static SetLong of(SetLong setLong1,SetLong setLong2,SetLong setLong3){
        return SetLong.of(SetLong.of(setLong1,setLong2),setLong3);
    }


    public static SetLong of(SetLong setLong1,SetLong setLong2){
        List<Long> allSetLong = Stream.of(setLong1, setLong2).flatMap(SetLong::stream).collect(Collectors.toList());
        return SetLong.of(allSetLong);
    }


    public static SetLong of(SetLong setLong){
        return SetLong.of(setLong.out());
    }


    /**
     * 从Stream 中 得到一个集合
     * @param stream 集合
     * @return SetLong
     */
    public static SetLong of(Stream<Long>stream){
        return of(stream.collect(Collectors.toSet()));
    }


    /**
     * 从Stream 中 得到一个集合
     * @param stream 集合
     * @param longGetter 数据提取器
     * @return SetLong
     * @param <B> 源数据类型
     */
    public static <B> SetLong of(Stream<B> stream, Function<B,Long> longGetter){
        return of(stream.map(longGetter).collect(Collectors.toSet()));
    }


    public static SetLong of(Set<Long> set){
        AssertUtil.assertMethodRequire(set,"set");
        return new SetLong(set);
    }

    public static SetLong of(Collection<Long> coll){
        return of(new HashSet<>(coll));
    }
    public static SetLong of(Collection<Long> coll, Predicate<Long> filter){
        return of(coll.stream().filter(filter).collect(Collectors.toSet()));
    }

    /**
     * 过滤内容
     * @param filter 过滤器
     * @return SetLong
     */
    public SetLong filter(Predicate<Long> filter){
        return of(value.stream().filter(filter).collect(Collectors.toSet()));
    }


    /**
     * 把空的数据过滤掉,只保留非空的
     * @return SetLong
     */
    public SetLong filterNonNull(){
        return filter(Objects::nonNull);
    }



    /**
     * 创建
     * @param longGetter
     * @param coll
     * @param <L>
     * @deprecated 请使用 {@link #of(Collection,Function)}
     * @return SetLong
     */
    public static <L> SetLong of(Function<L,Long> longGetter,Collection<L> coll){
        return of(coll.stream()
                .map(longGetter)
                .collect(Collectors.toSet()));
    }
    /**
     * 创建一个集合
     * @param coll 源集合
     * @param longGetter long的获取器
     * @param <L> 源集合的类型
     * @return SetLong
     */
    public static <L> SetLong of(Collection<L> coll,Function<L,Long> longGetter){
        return of(coll.stream()
                .map(longGetter)
                .collect(Collectors.toSet()));
    }
    /**
     * 创建一个集合
     * @param coll 源集合
     * @param longGetter long的获取器
     * @param filter 过滤器
     * @param <L> 源集合的类型
     * @return SetLong
     */
    public static <L> SetLong of(Collection<L> coll,Function<L,Long> longGetter,Predicate<Long> filter){
        return of(coll,longGetter).filter(filter);
    }

    /**
     * 创建一个集合
     * 先过滤再取值
     * @param coll 源集合
     * @param filter bean的测试器
     * @param longGetter long的获取器
     * @return SetLong
     * @param <L>  源集合的类型
     */
    public static <L> SetLong ofFilter(Collection<L> coll,Predicate<L> filter,Function<L,Long> longGetter){
        List<Long> longs = coll.stream().filter(filter).map(longGetter).collect(Collectors.toList());
        return of(longs);
    }






    /**
     * 创建一个集合
     * @param coll 源集合
     * @param longGetter long的获取器
     * @param filter 过滤器
     * @param filter2 过滤器2
     * @param <L> 源集合的类型
     * @return SetLong
     */
    public static <L> SetLong of(Collection<L> coll,Function<L,Long> longGetter,Predicate<Long> filter,Predicate<Long> filter2){
        return of(coll,longGetter,filter).filter(filter2);
    }
    /**
     * 创建
     * @param longGetter
     * @param coll
     * @param <L>
     * @return SetLong
     */
    public static <L,B> SetLong of(Collection<L> coll,Function<L,B> beanGetter,Function<B,Long> longGetter){
        return of(coll,beanGetter.andThen(longGetter));
    }
    /**
     * 创建
     * @param longGetter
     * @param coll
     * @param <L>
     * @return SetLong
     */
    public static <L,B,B2> SetLong of(Collection<L> coll,Function<L,B> beanGetter,Function<B,B2> beanGetter2,Function<B2,Long> longGetter){
        return of(coll,beanGetter.andThen(beanGetter2).andThen(longGetter));
    }


    /**
     * 创建
     * @param longGetter
     * @param coll
     * @param <L>
     * @param filter 过滤器
     * @deprecated 请使用 {@link #of(Collection,Function, Predicate)}
     * @return SetLong
     */
    @Deprecated
    public static <L> SetLong of(Function<L,Long> longGetter, Collection<L> coll, Predicate<L> filter){
        return of(coll.stream().filter(filter)
                .map(longGetter)
                .collect(Collectors.toSet()));
    }
    /**
     * 创建
     * @param longGetter
     * @param coll
     * @param <L>
     * @param filter 过滤器
     * @deprecated 请使用 {@link #of(Collection,Function, Predicate, Predicate)}
     * @return SetLong
     */
    @Deprecated
    public static <L> SetLong of(Function<L,Long> longGetter, Collection<L> coll, Predicate<L> filter,Predicate<Long> filter2){
        return of(coll.stream().filter(filter)
                .map(longGetter)
                .filter(filter2)
                .collect(Collectors.toSet()));
    }


    /**
     * 转换成数组
     * @return 数组
     */
    public Long[] toArray(){
        return value.toArray(new Long[0]);
    }




    /**
     * 与另外一个取交集
     * @param other 另外一个
     * @return 交集
     */
    public SetLong intersect(SetLong other){
        return SetLong.of(this.stream().filter(other::contains).collect(Collectors.toSet()));
    }


    /**
     * 将所有的集合取交集
     * @param setLongs 集合
     * @return 交集
     */
    public static Optional<SetLong> intersectAll(Collection<SetLong> setLongs){
        List<SetLong> setLongList = new ArrayList<>(setLongs);
        return setLongList.stream().findFirst().map(first-> setLongList.stream().skip(1).reduce(first, SetLong::intersect));
    }


    /**
     * 与另外一个取并集
     * @param others 另外一个
     * @return 并集
     */
    public SetLong union(SetLong... others){
        Set<Long> union = new HashSet<>(this.value);
        Stream.of(others).forEach(other->union.addAll(other.value));
        return SetLong.of(union);
    }


    /**
     * 将所有的集合取并集
     * @param setLongs 集合
     * @return 并集
     */
    public static SetLong unionAll(Collection<SetLong> setLongs){
        Set<Long> union = new HashSet<>();
        setLongs.forEach(setLong -> union.addAll(setLong.value));
        return SetLong.of(union);
    }






    /**
     * 根据Stream提取器来创建
     * @param longStreamFinder  long的stream获取器
     * @param coll 集合
     * @return SetLong
     * @param <L> 集合的类型
     */
    public static <L> SetLong ofStreamFinder(StreamFinder<L,Long> longStreamFinder, Collection<L> coll){
        return of(coll.stream().flatMap(longStreamFinder::get).collect(Collectors.toSet()));
    }




    /**
     * 根据Stream提取器来创建
     * @param longStreamFinder  long的stream获取器
     * @param coll 集合
     * @return SetLong
     * @param <L> 集合的类型
     */
    public static <L> SetLong ofStreamFinder(Collection<L> coll,StreamFinder<L,Long> longStreamFinder){
        return of(coll.stream().flatMap(longStreamFinder::get).collect(Collectors.toSet()));
    }


    /**
     * 复制出一个副本
     * @return
     */
    public SetLong copy(){
        return SetLong.of(out());
    }


    //----------------------------测试器----------------------------


    /**
     * 是不是存在
     * @param target
     * @return
     */
    public boolean contains(long target){
        return value.contains(target);
    }




    //---------------------------输出-----------------------------


    /**
     * 输出
     * @return
     */
    public Set<Long> out(){
        return value;
    }


    /**
     * 返回一个stream
     * @return
     */
    public Stream<Long> stream(){
        return value.stream();
    }


    /**
     * 判断是不是空的
     * @return
     */
    public boolean isEmpty(){
        return value.isEmpty();
    }

    /**
     * 判断是不是  非空。<BR>
     * 这里注意，对于  [null] 也代表是非空,<BR>
     * 我们的规范中，接受有  这样的数组 [null,""] 也就是list set map 等中不允许用 null，如果业务可能用就应该使用List<Optional<Bean>> 的方式<BR>
     * TODO 这段话要加入代码标准。
     * @return  空返回false 非空返回true
     */
    public boolean isNotEmpty(){
        return !this.isEmpty();
    }

}
