package com.mini.framework.util.qrcode;


import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Shape;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;

import javax.imageio.ImageIO;
import javax.imageio.stream.ImageOutputStream;
import javax.imageio.stream.MemoryCacheImageOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.util.asserts.AssertUtil;
import com.google.gson.Gson;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.Binarizer;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.EncodeHintType;
import com.google.zxing.LuminanceSource;
import com.google.zxing.MultiFormatReader;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.NotFoundException;
import com.google.zxing.Result;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.BufferedImageLuminanceSource;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;
import com.mini.framework.core.exception.HandleEnDeCodeException;
import com.mini.framework.core.exception.HandleIOException;

public class QrcodeUtil {
	
	private Logger logger = LogManager.getLogger(QrcodeUtil.class);
	
	private static HashMap<EncodeHintType,Object> basicParams = createParams();
	private static final String format = "png";
	
	private int width = 400;
	private int height = 400;
	private String charset = "UTF-8";
	
	public QrcodeUtil() {
		
	}
	public QrcodeUtil(int size) {
		this(size,size);
	}
	public QrcodeUtil(int width, int height) {
		super();
		this.width = width;
		this.height = height;
	}

	public byte[] createBytes(String content){
		AssertUtil.assertNotEmpty(content, "转维码的数据不能为空");
		MultiFormatWriter mfw = new MultiFormatWriter();
        ByteArrayOutputStream stream = new  ByteArrayOutputStream();
        BitMatrix bitMatrix;
		try {
			bitMatrix = mfw.encode(content, BarcodeFormat.QR_CODE, width, height,basicParams);
			MatrixToImageWriter.writeToStream(bitMatrix, format, stream );
			return stream.toByteArray();
		} catch (WriterException e) {
			throw new HandleEnDeCodeException(e,"把[%s]生成二维码时出错", content);
		} catch (IOException e) {
			throw new HandleIOException(e,"把[%s]生成二维码时出错", content);
		}finally{
			IOUtils.closeQuietly(stream);
		}
	}
	
	/**
	 * 尝试解析二维码，如果失败返回 空不要抛出异常
	 * @param picByte
	 * @return
	 */
	public String tryDecode(byte[] picByte){
		try{
			return decode(new ByteArrayInputStream(picByte));
		}catch(Throwable e){
			logger.debug("尝试解析二维码时失败",e);
			return null;
		}
	}

	public String decode(byte[] picByte){
		return decode(new ByteArrayInputStream(picByte));
	}
	
	
	/**
	 * 解析二维码
	 * @param picByte
	 * @return
	 */
	public String decode(InputStream is){
		try {
			BufferedImage bufferedImage = ImageIO.read(is);
			LuminanceSource source = new BufferedImageLuminanceSource(bufferedImage);
			Binarizer binarizer = new HybridBinarizer(source);
			BinaryBitmap bitmap = new BinaryBitmap(binarizer);
			HashMap<DecodeHintType, Object> decodeHints = new HashMap<DecodeHintType, Object>();
			decodeHints.put(DecodeHintType.POSSIBLE_FORMATS, Arrays.asList(BarcodeFormat.QR_CODE));
			// decodeHints.put(DecodeHintType.TRY_HARDER, Boolean.TRUE);
			decodeHints.put(DecodeHintType.CHARACTER_SET, charset);
			Result result = new MultiFormatReader().decode(bitmap, decodeHints);
			return result.getText();
		} catch (NotFoundException e) {
			// e ingore 这个框框架里的异常没有东西，qr框架没有设计好。
			throw new HandleEnDeCodeException("无法识别二维码");
		} catch (IOException e) {
			throw new HandleIOException(e,"解析二维码时流出错");
		}
	}
	
	private static HashMap<EncodeHintType, Object> createParams() {
        //定义二维码的参数
        HashMap<EncodeHintType,Object> params = new HashMap<>();
        //设置编码
        params.put(EncodeHintType.CHARACTER_SET, "utf-8");
        //设置纠错等级
        params.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.M);
        //设置边框
        params.put(EncodeHintType.MARGIN, 0);
		return params;
	}

	/**
	 * 把一个对象转成json再成二维码
	 * @param obj
	 * @return
	 */
	public byte[] createBytes(Object obj){
		AssertUtil.assertNotNull(obj, "转二维码的数据不能为空");
		String content = new Gson().toJson(obj);
		return createBytes(content);
	}
	
	
	//TODO 
    public static void insertImage(BufferedImage source, String imgPath, int x, int y) throws MalformedURLException, IOException {
        Image src = ImageIO.read(new URL(imgPath));
        int width = src.getWidth(null);
        int height = src.getHeight(null);
        Graphics2D graph = source.createGraphics();
        graph.drawImage(src, x, y, width, height, null);
        Shape shape = new RoundRectangle2D.Float(x, y, width, height, 10, 10);
        graph.setStroke(new BasicStroke(3f));
        graph.draw(shape);
        graph.dispose();
    }
    
    
    /**
     * 给图添加logo返回字节码。
     * @param target 目标图片。
     * @param sourse logo源
     * @param sizeRaito logo占比。这里会自动居中处理。
     * @return
     */
    public byte[] insertLogo(byte[] target,byte[] sourse, int sizeRaito){
//    	BufferedImage newImg = new BufferedImage(200,200, BufferedImage.TYPE_INT_ARGB);
		try {
			BufferedImage targetBi = ImageIO.read(new ByteArrayInputStream(target));
			BufferedImage sourseBi = ImageIO.read(new ByteArrayInputStream(sourse));
			int width = targetBi.getWidth();
			int height = targetBi.getHeight();
			int widthSize = width * sizeRaito/100;
			int widthBegin = (width- widthSize)/2;
			int heightSize = height * sizeRaito/100;
			int heightBegin =( height-heightSize)/2;
			BufferedImage newImg = new BufferedImage(width,height, BufferedImage.TYPE_3BYTE_BGR);
	    	int[] rgbArray = targetBi.getRGB(0, 0, width, height, null , 0, width);
	    	newImg.setRGB(0, 0, width, height, rgbArray , 0, width);
	    	int sourseWidth = sourseBi.getWidth();
	    	int sourseHeight = sourseBi.getHeight();
	    	for (int i = 0; i < widthSize; i++) {
	    		for (int j = 0; j < heightSize; j++) {
	    			int rgb = sourseBi.getRGB(i*sourseWidth/widthSize, j*sourseHeight/heightSize);
					newImg.setRGB(i+ widthBegin ,j+heightBegin, rgb );
				}
			}
	    	
	    	ByteArrayOutputStream stream = new ByteArrayOutputStream();
			ImageOutputStream output = new MemoryCacheImageOutputStream(stream );
			ImageIO.write(newImg, "jpg", output );
			return stream.toByteArray();
		} catch (IOException e) {
			throw new HandleIOException(e, "添加logo失败", sizeRaito);
		}
    }
	
}
