package com.mini.framework.util.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpRequest;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.config.RequestConfig.Builder;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.HandleEnDeCodeException;
import com.mini.framework.core.exception.HandleIOException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.params.MapParams;
import com.mini.framework.util.string.StringUtil;


/**
 * 简易http请求
 * 目前公用的http请求功能很齐全，但实际上我们要用的东西很少
 * 比如头部信息等，一般用不上，但又很纠结
 * 这里代理一个简易的http请求同时把异常公司的统一异常
 *
 * @author jayheo
 */
public class SimpleHttpUtil {

    protected final static Logger logger = LogManager.getLogger(SimpleHttpUtil.class);

    protected final static String METHOD_POST_NAME = "POST";
    protected final static String METHOD_GET_NAME = "GET";
    protected final static String CHARSET_UTF8 = "utf-8";

    protected static RequestConfig defaultRequestConfig = initDefaultRequestConfig();

    public static String simpleRequest(String url, String method, Map<String, String> params, Map<String, String> headers) {
        byte[] byteBuf = simpleRequest(true, url, method, params, null, null);
        return StringUtil.toUtf8String(byteBuf);
    }

    private static RequestConfig initDefaultRequestConfig() {
        Builder builder = RequestConfig.custom();
        //连接建立时间，三次握手完成时间
        builder.setConnectTimeout(6000);
        //httpclient使用连接池来管理连接，这个时间就是从连接池获取连接的超时时间，可以想象下数据库连接池
        builder.setConnectionRequestTimeout(60000);
        //数据传输过程中数据包之间间隔的最大时间
        builder.setSocketTimeout(12000);
        builder.setMaxRedirects(5);
        return builder.build();
    }

    public static String simpleRequest(String url, String method, String body, Map<String, String> headers) {
        byte[] byteBuf = simpleRequest(false, url, method, null, body, headers);
        return StringUtil.toUtf8String(byteBuf);
    }


    private static byte[] simpleRequest(boolean keyAndValue, String url, String method, Map<String, String> params, String body, Map<String, String> headers) {
        logger.debug(String.format("http request,keyAndValue:%s,method:%s,url:%s,params:%s,body:%s,header:%s", keyAndValue, method, url, params, body, headers));
        CloseableHttpClient closeableHttpClient = HttpClientBuilder.create().build();
        int status = 0;
        byte[] byteBuf = null;
        String result = null;
        CloseableHttpResponse resp = null;
        HttpContext context = new BasicHttpContext();
        try {
            if (keyAndValue) {//直接发post请求，没key value
                HttpUriRequest request = new SimpleRequest(method, url);
                if (params != null) {
                    for (Entry<String, String> entry : params.entrySet()) {
                        context.setAttribute(entry.getKey(), entry.getValue());
                    }
                }
                buildHead(headers != null, request, headers);
                resp = closeableHttpClient.execute(request, context);
            } else {//有key value
                HttpPost httpPost = new HttpPost(url);
                httpPost.setConfig(defaultRequestConfig);
                HttpEntity entity = new StringEntity(body, CHARSET_UTF8);
                httpPost.setEntity(entity);
                buildHead(headers != null, httpPost, headers);
                resp = closeableHttpClient.execute(httpPost, context);
            }
            status = resp.getStatusLine().getStatusCode();
            InputStream content = resp.getEntity().getContent();
            byteBuf = IOUtils.toByteArray(content);
            //StringWriter sw = new StringWriter();
            //IOUtils.copy(content, sw, CHARSET_UTF8);
            result = StringUtil.toUtf8String(byteBuf);
            logger.debug(String.format("http status code:%s,result:%s", status, result));
        } catch (IOException e) {
            throw new HandleIOException(e, "在发http:[%s]请求时出错",url);
        } finally {
            IOUtils.closeQuietly(resp);
        }

        if (200 <= status && 300 > status) {
            return byteBuf;
        } else {
            throw new HandleIOException("http-error,status:%s,url:%s", status, url);
        }
    }

    /**
     * 简单post请求,直接发报文
     */
    public static String simplePost(String url, String body) {
        return simpleRequest(url, METHOD_POST_NAME, body, null);
    }

    /**
     * 简单post请求,直接发报文
     */
    public static String simplePost(String url, String body, Map<String, String> headers) {
        return simpleRequest(url, METHOD_POST_NAME, body, headers);
    }

    /**
     * 简单post请求,直接发报文
     * 返回字节
     *
     * @param url
     * @param body
     * @return
     */
    public static byte[] simpleBytePost(String url, String body) {
        byte[] byteBuf = simpleRequest(false, url, METHOD_POST_NAME, null, body, null);
        return byteBuf;
    }


    /**
     * 简单post请求
     */
    public static String simplePost(String url, Map<String, String> params) {
        return simpleRequest(url, METHOD_POST_NAME, params, null);
    }

    /**
     * 简单get请求
     */
    public static String simpleGet(String url, Map<String, String> params) {
    	url = assemGetRequestEncode(url,Optional.ofNullable(params));
        return simpleRequest(url, METHOD_GET_NAME, (Map<String, String>)null, null);
    }

    /**
     * 拼装Get请求
     * @param url
     * @param paramsOptional
     * @return
     */
    protected static String assemGetRequestEncode(String url,Optional<Map<String, String>> paramsOptional) {
        if (!paramsOptional.isPresent()) {
            return url;
        }else{
        	for (Entry<String, String> entry : paramsOptional.get().entrySet()) {
        		try {
        			if (entry.getValue() == null) continue;
        			String encode = URLEncoder.encode(entry.getValue(), CHARSET_UTF8);
        			entry.setValue(encode);
        		} catch (UnsupportedEncodingException e) {
        			throw new HandleEnDeCodeException(e, "getRequestEncode转码时出现异常,,entry:%s", entry);
        		}
        	}
        	MapParams params = MapParams.build();
        	params.putAll(paramsOptional.get());
        	return params.toUrlParams(url);
        }
    }

    /**
     * 简单post请求
     */
    public static String simplePost(String url) {
        return simplePost(url, (Map<String, String>) null);
    }

    /**
     * 简单get请求
     */
    public static String simpleGet(String url) {
        return simpleGet(url, null);
    }

    public static void buildHead(boolean explain, HttpRequest request, Map<String, String> headers) {
        if (explain) headers.forEach((k, v) -> request.setHeader(k, v));
    }

    protected static class SimpleRequest extends HttpRequestBase {
        public SimpleRequest(String method, String url) {
            this.method = method;
            this.setConfig(defaultRequestConfig);
            try {
                super.setURI(new URI(url));
            } catch (URISyntaxException e) {
                throw new BadReqException(Status.BadReq.illUrlSyntax, "url语法错误:%s", url);
            }
        }

        private String method;

        @Override
        public String getMethod() {
            return method;
        }

    }

}
