package com.mini.framework.util.report.statistics.protocol;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.exception.standard.CustomExceptionSupplier;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateRange;
import com.mini.framework.util.date.DateUtil;
import com.mini.framework.util.object.SetUsefulObject;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * 时间的区间的范围
 * @author jayheo
 */
public class TimeRegionRange extends SetUsefulObject<TimeRegionRange> {

    /**
     * 区域的 范围/跨度
     */
    private TimeRegionUnit unit;

    /**
     * 区域的时间起点
     */
    private Date offset;
    /**
     * TODO 需要添加文档说明这是什么意思
     */
    private Date offsetAllUnit;

    /**
     * offset 的简单字符串
     */
    private String offsetSimpleString;
    
    public TimeRegionRange(){}

    public static TimeRegionRange create(TimeRegionUnit unit, Date offset) {
        return create(unit, offset,error->new ServerException(Status.Server.programConfigJava,error));
    }
    public static <E extends CustomException> TimeRegionRange create(TimeRegionUnit unit, Date offset, CustomExceptionSupplier<E> exceptionSupplier) throws E{
        TimeRegionRange instance = new TimeRegionRange();
        instance.unit = unit;
        instance.offset = offset;
        // 检查 是不是合法的
        instance.validateOffsetRightPoint(exceptionSupplier);
        return instance.mapperShows();

    }

    public static TimeRegionRange createAllUnit(Date offsetAllUnit) {
        TimeRegionRange range = create(TimeRegionUnit.all, new Date(0));
        range.offsetAllUnit = offsetAllUnit;
        return range;
    }

    /**
     * 当前这个范围的跨度 单位是毫秒
     * @return
     */
    public long timeScopeCrossMilli(){
        return mapperToDate().getTime() - offset.getTime();
    }


    /**
     * 映射要显示的内容
     * @return
     */
    public TimeRegionRange mapperShows(){
        offsetSimpleString = DateUtil.toMilSecondsString(offset);
        return this;
    }

    public DateRange showDateRange(){
        if(unit.equalType(TimeRegionUnit.all)){
            //在现有的数据中加100年。
            return DateRange.create(offsetAllUnit,TimeRegionUnit.year.addTimeUnit(offsetAllUnit,100));
        }else{
            return DateRange.create(offset,mapperToDate());
        }
    }

    /**
     * 判断两个range是不是相同
     * @param other
     * @return
     */
    public boolean same(TimeRegionRange other){
        return offset.equals(other.offset) && unit.equals(other.unit);
    }


    /**
     * 生成一个随机的排序器
     * @return
     */
    public static Comparator<TimeRegionRange> randomSorter(){
        //一个万以下的最大质数
        int primeNumber = 9973;
        return Comparator.comparing(range -> range.offset.getTime() * range.hashCode()/ primeNumber);
    }

    /**
     * 根据开始时间得到升序的排序器
     * @return
     */
    public static Comparator<TimeRegionRange> offsetUpperSorter(){
        return Comparator.comparing(TimeRegionRange::getOffset);
    }


    /**
     * 当前对应父级的区间范围
     * @param superUnit
     * @return
     */
    public TimeRegionRange findSuperRegionRange(TimeRegionUnit superUnit){
        unit.validateCanAccumulateTo(superUnit);
        return superUnit.nearFenceRange(offset);
    }


    /**
     * 检验时间在不在正确的点上
     * @param <E>
     * @param exceptionSupplier
     * @throws E
     * @return
     */
    public <E extends CustomException> TimeRegionRange validateOffsetRightPoint(CustomExceptionSupplier<E> exceptionSupplier)throws E{
        exceptionSupplier.applyAssert(offset!=null,"offset不能为空");
        exceptionSupplier.applyAssert(unit!=null,"unit不能为空");
        Date rightPoint = unit.nearLazyFenceRangeDate(offset);
        exceptionSupplier.applyAssert(rightPoint.equals(offset)
                ,"range中unit:[%s]当前offset:[%s]与正确的offset:[%s]不一致",unit,offset.getTime(),rightPoint.getTime());
        return this;
    }



    public Optional<List<TimeRegionRange>> mapperComposeElementFences(){
        return unit.getComposeElement()
                .map(unit->unit.regionFences(offset,mapperToDate()));
    }

    /**
     * 查到第一个。
     * @return
     */
    public Optional<TimeRegionRange> showComposeElement(){
        return mapperComposeElementFences().flatMap(list->list.stream().findFirst());
    }

    public Date mapperToDate(){
        return unit.addTimeUnit(offset,1);
    }


    public boolean matchUnit(TimeRegionUnit unit) {
        return this.unit.equals(unit);
    }

    /**
     * 比较两个range是不是相同
     * @param other
     * @return
     */
    public boolean match(TimeRegionRange other){
        return matchUnit(other.unit) && offset.equals(other.offset);
    }
    
    public TimeRegionRange nextRegionRange(int amount){
        return TimeRegionRange.create(unit,unit.addTimeUnit(offset,amount));
    }
    
    
    public TimeRegionRange nextRegionRange(){
        return nextRegionRange(1);
    }
    
    public TimeRegionRange previousRegionRange(){
        return nextRegionRange(-1);
    }
    
    public TimeRegionRange nextSameRegionRange(TimeRegionUnit superUnit, int amount){
        unit.validateCanAccumulateTo(superUnit);
        Date superOffset = superUnit.nearLazyFenceRangeDate(offset);
        int itemIndex = unit.regionFences(superOffset, offset).size();
        Date otherSuperOffset = superUnit.addTimeUnit(superOffset, amount);
        Date otherSameDate = this.unit.addTimeUnit(otherSuperOffset, itemIndex);
        return TimeRegionRange.create(unit,otherSameDate);
    }


    /**
     * 检查目标范围是不是父范围
     * @param superScope 父范围
     * @return
     */
    public boolean matchSuperScope(TimeRegionRange superScope){
        AssertUtil.assertMethodRequire(superScope,"superScope");
        unit.validateCanAccumulateTo(superScope.unit);
        return matchSuperScope(showDateRange());

    }

    /**
     * 检查目标范围是不是父范围
     * @param superScope
     * @return
     */
    public boolean matchSuperScope(DateRange superScope){
        return !this.offset.before(superScope.getMinDate())
                &&!superScope.getMaxDate().before(this.mapperToDate());
    }


    /**
     * 把自己分解成多个自然时间 要保证尽量少的分段数
     * @return
     */
    public List<TimeRegionRange> splitRangeLessPoint(TimeRegionUnit minUnit,TimeRegionUnit maxUnit){
        return splitRangeLessPoint(showDateRange(),minUnit,maxUnit , CustomExceptionSupplier.fromFunction(message->new ServerException("%s 出现bug需要检查程序",message)));
    }


    /**
     * 把一个时间范围分解成多个自然时间。
     * @param scopeRange
     * @param exceptionSupplier 如果出现不能完整切分的时间，就会要报错
     * @return
     */
    public static List<TimeRegionRange> splitRangeLessPoint(DateRange scopeRange,TimeRegionUnit minUnit,TimeRegionUnit maxUnit, CustomExceptionSupplier exceptionSupplier){
        return splitRangeLessPointHandleChip(scopeRange,minUnit,maxUnit, restChip->{
            throw exceptionSupplier.supply("时间区间[%s,%s)找不到细分自然时间区间",restChip.getMinDate().getTime(),restChip.getMinDate().getTime());
        });

    }


    /**
     * 把一个时间范围分解成多个自然时间。
     * @param scopeRange
     * @param maxUnit 允的最大单元
     * @param onFoundRestChip 如果出现不能完整切分的时间 就调用这个方法。
     * @return
     */
    public static List<TimeRegionRange> splitRangeLessPointHandleChip(DateRange scopeRange,TimeRegionUnit minUnit,TimeRegionUnit maxUnit, Consumer<DateRange> onFoundRestChip){
        if(scopeRange.haveTimeScope()){
            Date rangeMinDate = scopeRange.getMinDate();
            Date rangeMaxDate = scopeRange.getMaxDate();
            //从大到小(时间范围) 从左到右 找出一个尽量大子区间。
            Optional<TimeRegionRange> subRegionRangeOptional = TimeRegionUnit.showElementUsefulDownSort(minUnit,maxUnit).stream()
                    .map(unit -> TimeRegionRange.create(unit, unit.nearGreedyFenceRangeDate(rangeMinDate)))
                    .filter(range -> range.matchSuperScope(DateRange.create(rangeMinDate, rangeMaxDate)))
                    .findFirst();

            List<TimeRegionRange> allRange = new ArrayList<>();

            DateRange discardRange = subRegionRangeOptional.map(TimeRegionRange::showDateRange).orElse(scopeRange);
            //切分时间段成为三段: 前段 + 自己 + 后段 ,然后加到一起。
            List<TimeRegionRange> beforeRanges = splitRangeLessPointHandleChip(DateRange.create(rangeMinDate,discardRange.getMinDate()),minUnit,maxUnit,onFoundRestChip);
            List<TimeRegionRange> afterRanges = splitRangeLessPointHandleChip(DateRange.create(discardRange.getMaxDate(),rangeMaxDate),minUnit,maxUnit,onFoundRestChip);
            allRange.addAll(beforeRanges);
            subRegionRangeOptional.ifPresent(allRange::add);
            if(!subRegionRangeOptional.isPresent()){
                onFoundRestChip.accept(scopeRange);
            }
            allRange.addAll(afterRanges);
            return allRange;
        }else{
            return new ArrayList<>();
        }

    }


    /**
     * 根据子元素查询出区域栅栏
     * @param elementUnit
     * @return
     */
    public List<TimeRegionRange> findElementRegionFences(TimeRegionUnit elementUnit){
        AssertUtil.assertMethodRequire(elementUnit,"elementUnit");
        elementUnit.validateCanAccumulateTo(unit);
        return unit.findElementRegionFences(TimeRegionRange.create(elementUnit,offset));
    }
    
    public String unitOffsetUnionString(){
        return String.format("%s-%s", unit, DateUtil.toSecondsString(offset));
    }

    public Date getOffset() {
        return offset;
    }

    public TimeRegionRange setOffset(Date offset) {
        this.offset = offset;
        return this;
    }

    public TimeRegionUnit getUnit() {
        return unit;
    }

    public TimeRegionRange setUnit(TimeRegionUnit unit) {
        this.unit = unit;
        return this;
    }

    public Date getOffsetAllUnit() {
        return offsetAllUnit;
    }

    public void setOffsetAllUnit(Date offsetAllUnit) {
        this.offsetAllUnit = offsetAllUnit;
    }

    public String getOffsetSimpleString() {
        return offsetSimpleString;
    }

    public TimeRegionRange setOffsetSimpleString(String offsetSimpleString) {
        this.offsetSimpleString = offsetSimpleString;
        return this;
    }


    @Override
    protected Function<TimeRegionRange, Object>[] showHashCodeCells() {
        Function<TimeRegionRange, Object> fn1 = TimeRegionRange::getUnit;
        Function<TimeRegionRange, Object> fn2 = TimeRegionRange::getOffset;
        Function<TimeRegionRange, Object> fn3 = TimeRegionRange::getOffsetSimpleString;
        return new Function[]{fn1,fn2,fn3};
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
